/*
 *  This file is part of the Open C Platform (OCP) Library. OCP is a
 *  portable library for development of Data Communication Applications.
 * 
 *  Copyright (C) 1995 Neda Communications, Inc.
 * 	Prepared by Mohsen Banan (mohsen@neda.com)
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Library General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.  This library is
 *  distributed in the hope that it will be useful, but WITHOUT ANY
 *  WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 *  License for more details.  You should have received a copy of the GNU
 *  Library General Public License along with this library; if not, write
 *  to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 *  USA.
 * 
 */

/*
 * Author: Derrell Lipman
 * History:
 *
 */

/*
 * Module Management
 */


#include "estd.h"
#include "rc.h"
#include "queue.h"
#include "tmr.h"
#include "mm.h"
#include "mmlocal.h"


/*
 * Static Variables
 *
 * CAUTION: This may not be thread-safe.  The remainder of this module
 *	    should be thread-safe.
 */
static QU_Head		applicationEntityInstances =
				QU_INITIALIZE(applicationEntityInstances);


/*
 * Forward declarations.
 */

static ReturnCode
generateEvent(MM_EventType eventType,
	      ManagableObject * pObject,
	      OS_Uint16 extraMemory);


static ReturnCode
timerExpiry(void * hTimer,
	    void * hUserData1,
	    void * hUserData2);


/*
 * MM_entityInit()
 *
 * Initialize the Module Management Entity module.
 *
 * Parameters:
 *
 *   pApplicationEntityInstanceName --
 *     The name of the application entity instance in which the module
 *     is located.  Note that if the same application is running in
 *     more then one instance, the instance name must be unique in
 *     each one.
 *
 *   pfRegisterApplication --
 *     Pointer to a function which will register this application
 *     entity instance with an independent agent.  If no independent
 *     agent exists, this pointer should be NULL.
 *
 *   phApplicationEntityInstance --
 *     Pointer to a location to put the handle for this application entity
 *     instance.  The handle must be passed to MM_registerModule() in order to
 *     associate the module with the application entity instance.
 */
ReturnCode
MM_entityInit(char * pApplicationEntityInstanceName,
	      ReturnCode (* pfRegisterApplication)(char *
						   pAppEntityInstanceName),
	      void ** phApplicationEntityInstance)
{
    ReturnCode 			rc;
    ApplicationEntityInstance *	pApplicationEntityInstance;

    /* Allocate an Application structure */
    if ((pApplicationEntityInstance =
	 OS_alloc(sizeof(ApplicationEntityInstance))) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_entityInit() "
			"out of memory getting application structure.\n"));
    }

    /* Save the application entity instance name */
    pApplicationEntityInstance->pName = pApplicationEntityInstanceName;

    /* Initialize the queue of modules for this application instance */
    QU_INIT(&pApplicationEntityInstance->modules);

    /* Initialize the pending-event queue */
    QU_INIT(&pApplicationEntityInstance->pendingEvents);

    /* Initialize the destination queue */
    QU_INIT(&pApplicationEntityInstance->destinations);

    /* Initialize the queue pointers */
    QU_INIT(pApplicationEntityInstance);

    /* If an independent agent registration pointer was provided... */
    if (pfRegisterApplication != NULL)
    {
	/* ... then register this application entity instance with the agent */
	if ((rc =
	     (* pfRegisterApplication)(pApplicationEntityInstanceName)) !=
	    Success)
	{
	    return FAIL_RC(rc,
			   ("MM_entityInit(): register independent agent\n"));
	}
    }

    /* Insert this application entity instance onto the queue of instances */
    QU_INSERT(pApplicationEntityInstance, &applicationEntityInstances);

    /* Give 'em what they came for */
    *phApplicationEntityInstance = pApplicationEntityInstance;
    
    return Success;
}




/*
 * MM_registerModule()
 *
 * Allocate management resources for a code module or protocol layer.
 *
 * Parameters:
 *
 *   hApplicationEntityInstance --
 *     Handle, previously provided by MM_entityInit(), indicating to
 *     which application entity instance this module is attached.
 *
 *   pModuleName --
 *     The name of the module for which managable entities are to be
 *     registered.
 *
 *   phModule --
 *     Pointer to a handle.  The handle is generated by this function.
 *     Future requests to register a managable object will use this
 *     handle.
 */
ReturnCode
MM_registerModule(void * hApplicationEntityInstance,
		  char * pModuleName,
		  void ** phModule)
{
    Module * 			pModule;
    ApplicationEntityInstance *	pApplicationEntityInstance;

    /* Get a pointer to the application entity instance structure */
    pApplicationEntityInstance = hApplicationEntityInstance;
    
    /* Make sure this module name doesn't already exist */
    if ((pModule = mm_findModule(pModuleName,
				 pApplicationEntityInstance)) != NULL)
    {
	return FAIL_RC(MM_RC_NameAlreadyInUse,
		       ("MM_registerModule(): name (%s) already in use\n",
			pModuleName));
    }

    /* Allocate a new module structure */
    if ((pModule = OS_alloc(sizeof(Module))) == NULL)
    {
	return FAIL_RC(ResourceError, ("MM_registerModule() out of memory\n"));
    }

    /* Assign this module's name */
    if ((pModule->pName = OS_allocStringCopy(pModuleName)) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_registerModule() out of memory on module name\n"));
    }

    /* Keep track of which application entity instance this module is in */
    pModule->pApplicationEntityInstance = pApplicationEntityInstance;

    /* Initialize the queue pointers */
    QU_INIT(pModule);

    /* Initialize the queue of managable objects */
    QU_INIT(&pModule->managableObjects);

    /* Insert this module on the application entity instance queue */
    QU_INSERT(pModule, &pApplicationEntityInstance->modules);

    /* Give 'em what they came for */
    *phModule = pModule;

    return Success;
}


/*
 * MM_registerManagableObject()
 *
 * Register a managable object for use by the specified module.
 *
 * Parameters:
 *
 *   hModule --
 *     A module handle previously returned by MM_registerModule().
 *
 *   ManagableObjectType --
 *     The type of managable object being registered.
 *
 *   pManagableObjectName --
 *     The name of the managable object being registered.  This name must be
 *     unique within the scope of this module.
 *
 *   pDescription --
 *     A descriptive identification string which will be passed to the
 *     module management agent when an event is raised.
 *
 *   accessByName --
 *     Bits identifying whether the byName functions allow read
 *     and/or write access to this managable object.
 *  
 *   initialNotifyMask --
 *     Bits identifying the urgency of an event rasied for this managable
 *     object.  Multiple bits may be specified, but this use is discouraged.
 *
 *   pfValueChanged --
 *     Pointer to a function to be called when ever the value of a
 *     Counter, Guage, or String object is modified.  If the managable
 *     object type is not Counter, Guage, or String, this parameter
 *     should be set to NULL.  If no notification of value change is
 *     desired, this parameter may be set to NULL.
 *
 *   hUserData --
 *     This value will be passed back to the function pointed to by
 *     pfValueChanged when that function is called.  It may be used for any
 *     purpose desired by the user.
 *
 *   phManagableObject --
 *     Pointer to a handle.  The handle is generated by this function.  Future
 *     requests to set thresholds, modify values, etc. will require use of
 *     this handle.
 */
ReturnCode
MM_registerManagableObject(void * hModule,
			   MM_ManagableObjectType managableObjectType,
			   char * pManagableObjectName,
			   char * pDescription,
			   OS_Uint8 accessByName,
			   OS_Uint32 initialNotifyMask,
			   void (* pfValueChanged)(void * hUserData,
						   void * hManagableObject,
						   MM_Value * pValue),
			   void * hUserData,
			   void ** phManagableObject)
{
    Module *		pModule = hModule;
    ManagableObject * 	pObject;

    /* Make sure this module name doesn't already exist */
    if ((pObject =
	 mm_findObject(pManagableObjectName, pModule)) != NULL)
    {
	return FAIL_RC(MM_RC_NameAlreadyInUse,
		       ("MM_registerManagableObject(): "
			"name (%s) already in use\n",
			pManagableObjectName));
    }

    /* Allocate a new managable object structure */
    if ((pObject = OS_alloc(sizeof(ManagableObject))) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_registerManagableObject() out of memory\n"));
    }

    /* Assign this module's name */
    if ((pObject->pName = OS_allocStringCopy(pManagableObjectName)) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_registerManagableObject(): "
			"out of memory on object name\n"));
    }

    /* Keep track of the identification message */
    if ((pObject->pDescription = OS_allocStringCopy(pDescription)) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_registerManagableObject(): "
			"out of memory on identification message\n"));
    }

    /* Keep track of the initial notification types */
    pObject->notificationTypes = initialNotifyMask;

    /* Keep track of which module this managable object is associated with */
    pObject->pModule = pModule;

    /* Keep track of the function to call upon value change */
    pObject->pfValueChanged = pfValueChanged;

    /* Keep track of user's handle to be passed to the value change function */
    pObject->hUserData = hUserData;

    /* Keep track of the object type */
    pObject->objectType = managableObjectType;

    /* Keep track of who can access this object */
    pObject->accessByName = accessByName;

    /* See what object type this is */
    switch(managableObjectType)
    {
    case MM_ManagableObjectType_CounterSigned:
    case MM_ManagableObjectType_GaugeSigned:
	/*
	 * Initialize the value and thresholds (min doesn't apply to counter)
	 */
	pObject->valueType = ValueType_Sint32;
	pObject->value.sint32 = 0;
	pObject->minThreshold.sint32 = OS_MIN_SINT32;
	pObject->maxThreshold.sint32 = OS_MAX_SINT32;
	pObject->minAbsolute.sint32 = OS_MIN_SINT32;
	pObject->maxAbsolute.sint32 = OS_MAX_SINT32;
	break;
	
    case MM_ManagableObjectType_CounterUnsigned:
    case MM_ManagableObjectType_GaugeUnsigned:
	/*
	 * Initialize the value and thresholds (min doesn't apply to counter)
	 */
	pObject->valueType = ValueType_Uint32;
	pObject->value.uint32 = 0;
	pObject->minThreshold.uint32 = OS_MIN_UINT32;
	pObject->maxThreshold.uint32 = OS_MAX_UINT32;
	pObject->minAbsolute.uint32 = OS_MIN_UINT32;
	pObject->maxAbsolute.uint32 = OS_MAX_UINT32;
	break;

    case MM_ManagableObjectType_String:
	/* Initialize the value */
	pObject->valueType = ValueType_String;

	/* Allocate an empty string */
	if ((pObject->value.string = OS_alloc(1)) == NULL)
	{
	    return FAIL_RC(ResourceError,
			   ("MM_registerManagableObject() "
			    "Out of memory allocating for string value\n"));
	}

	/* It's just an empty string. */
	*pObject->value.string = '\0';
	break;

    case MM_ManagableObjectType_Timer:
	/* Initialize the timer handle (not actually a value) */
	pObject->valueType = ValueType_Handle;
	pObject->value.handle = NULL;
	break;

    case MM_ManagableObjectType_Log:
	pObject->valueType = ValueType_Handle;
	pObject->value.handle = NULL;
	break;

	/* No default so the compiler will catch it if we add more types */
    }

    /* Initialize the queue pointers */
    QU_INIT(pObject);

    /* Initialize this object onto the module's managable object queue */
    QU_INSERT(pObject, &pModule->managableObjects);

    /* Generate an event indicating that managable objects have been added */
    generateEvent(MM_EventType_ManagableObjectChange,
		  pObject, 0);

    /* Give 'em what they came for */
    *phManagableObject = pObject;

    return Success;
}


/*
 * MM_registerDestination()
 *
 * Register a new destination to which events may be sent.  A
 * destination is a place where an event is sent.  Destinations may
 * send event messages to a log file, send them via email, etc.
 *
 *
 * Parameters:
 *
 *   hApplicationEntityInstance --
 *     Handle, previously provided by MM_entityInit(), indicating to
 *     which application entity instance this module is attached.
 *
 *   pDestinationName --
 *     The name of the destination being registered.
 *
 *   pDestinationId --
 *     Identification string describing this destination.
 *
 *   accessByName --
 *     Bits identifying whether this destination's notify and event
 *     masks may be modified via the byName functions.
 *  
 *   initialNotifyMask --
 *     Bits specifying that events of notification types included in this mask
 *     are to be sent to this destination (in addition, possibly, to other
 *     destinations).
 *
 *   initialEventMask --
 *     Bits specifying that event types included in this mask are to
 *     be sent to this destination (in addition, possibly, to other
 *     destinations).
 *
 *   pfAlert --
 *     Pointer to a function which will be called when events are
 *     destined to this registered destination.
 *
 *     When the function pointed to by this parameter is ultimately
 *     called, it will be passed a set of zero or more optional
 *     parameters which are specific to the type of event which has
 *     been raised.  See the comments associated with the definition
 *     of MM_EventType.
 *
 *   phDestination
 *     Pointer to a handle.  The handle is generated by this function.  Future
 *     requests to modify the notification mask for this destination will
 *     require use of this handle.
 */
ReturnCode
MM_registerDestination(void * hApplicationEntityInstance,
		       char * pDestinationName,
		       char * pDescription,
		       OS_Uint8 accessByName,
		       OS_Uint32 initialNotifyMask,
		       OS_Uint32 initialEventMask,
		       void (* pfAlert)(char * pDestinationName,
					char * pApplicationEntityInstanceName,
					char * pModuleName,
					char * pManagableObjectName,
					char * pManagableObjectDescription,
					MM_EventType eventType,
					...),
		       void ** phDestination)
{
    Destination *	pDestination;
    ApplicationEntityInstance *	pApplicationEntityInstance;

    /* Get a pointer to the application entity instance structure */
    pApplicationEntityInstance = hApplicationEntityInstance;
    
    /* Make sure this destination name doesn't already exist */
    if ((pDestination =
	 mm_findDestination(pDestinationName,
			    pApplicationEntityInstance)) != NULL)
    {
	return FAIL_RC(MM_RC_NameAlreadyInUse,
		       ("MM_registerDestination(): "
			"name (%s) already in use\n",
			pDestinationName));
    }

    /* Allocate a new destination structure */
    if ((pDestination = OS_alloc(sizeof(Destination))) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_registerDestination() "
			"out of memory on Destination structure\n"));
    }

    /* Keep track of the name of this destination */
    if ((pDestination->pName = OS_allocStringCopy(pDestinationName)) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_registerDestination() "
			"out of memory on destination name"));
    }

    /* Keep track of the description of this destination */
    if ((pDestination->pDescription =
	 OS_allocStringCopy(pDescription)) == NULL)
    {
	return FAIL_RC(ResourceError,
		       ("MM_registerDestination() "
			"out of memory on destination description"));
    }

    /* Keep track of the access mode for this destination */
    pDestination->accessByName = accessByName;

    /* Keep track of the destination's call-back function */
    pDestination->pfAlert = pfAlert;

    /* Keep track of the initial notify mask */
    pDestination->notifyMask = initialNotifyMask;

    /* Keep track of the initial event mask */
    pDestination->eventMask = initialEventMask;

    /* Initialize the queue pointers */
    QU_INIT(pDestination);

    /* Insert this destination on the queue of destinations */
    QU_INSERT(pDestination, &pApplicationEntityInstance->destinations);

    /* Give 'em what they came for */
    *phDestination = pDestination;

    return Success;;
}



/*
 * MM_getThresholdByHandle()
 *
 * Get the maximum or minimum threshold value for a managable object.
 *
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously returned by
 *     MM_registerManagableObject().
 *
 *   thresholdType --
 *     Indication of whether the threshold to be set is a Maximum threshold or
 *     a Minimum threshold.
 *
 *   pValue --
 *     Pointer to a location in which current value of the threshold
 *     should be placed.  The pointer must be to the proper type for
 *     the managable object.  See the "Value Type" comments with the
 *     declaration of ManagableObjectType.
 *
 *
 * Note:
 *   Thresholds are only applicable to certain managable object types, such as
 *   Counters and Gauges.
 */
ReturnCode
MM_getThresholdByHandle(void * hManagableObject,
			MM_ThresholdType thresholdType,
			MM_Value * pValue)
{
    ManagableObject *	pObject = hManagableObject;

    /* Make sure this object type supports thresholds */
    switch(pObject->objectType)
    {
    case MM_ManagableObjectType_GaugeSigned:
    case MM_ManagableObjectType_GaugeUnsigned:
	break;

    case MM_ManagableObjectType_CounterSigned:
    case MM_ManagableObjectType_CounterUnsigned:
	if (thresholdType == MM_ThresholdType_MinimumThreshold ||
	    thresholdType == MM_ThresholdType_MinimumAbsolute)
	{
	    /* Minimum thresholds not supported for these types */
	    return FAIL_RC(MM_RC_MinimumThresholdNotSupported,
			   ("MM_getThreshold(MINIMUM) "
			    "not supported for type %d\n",
			    pObject->objectType));
	}
	break;
	
    case MM_ManagableObjectType_String:
    case MM_ManagableObjectType_Timer:
    case MM_ManagableObjectType_Log:
	/* Thresholds not supported for these types */
	return FAIL_RC(MM_RC_ThresholdNotSupported,
		       ("MM_getThreshold() not supported for type %d\n",
			pObject->objectType));
	break;

	/* No default so the compiler will catch it if we add more types */
    }

    /* See whether we're setting the minimum or maximum threshold */
    switch(thresholdType)
    {
    case MM_ThresholdType_MinimumThreshold:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    pValue->type = MM_ValueType_SignedInt;
	    pValue->un.signedInt = pObject->minThreshold.sint32;
	    break;
	    
	case ValueType_Uint32:
	    pValue->type = MM_ValueType_UnsignedInt;
	    pValue->un.unsignedInt = pObject->minThreshold.uint32;
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;
	
    case MM_ThresholdType_MaximumThreshold:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    pValue->type = MM_ValueType_SignedInt;
	    pValue->un.signedInt = pObject->maxThreshold.sint32;
	    break;
	    
	case ValueType_Uint32:
	    pValue->type = MM_ValueType_UnsignedInt;
	    pValue->un.unsignedInt = pObject->maxThreshold.uint32;
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;

    case MM_ThresholdType_MinimumAbsolute:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    pValue->type = MM_ValueType_SignedInt;
	    pValue->un.signedInt = pObject->minAbsolute.sint32;
	    break;
	    
	case ValueType_Uint32:
	    pValue->type = MM_ValueType_UnsignedInt;
	    pValue->un.unsignedInt = pObject->minAbsolute.uint32;
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;
	
    case MM_ThresholdType_MaximumAbsolute:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    pValue->type = MM_ValueType_SignedInt;
	    pValue->un.signedInt = pObject->maxAbsolute.sint32;
	    break;
	    
	case ValueType_Uint32:
	    pValue->type = MM_ValueType_UnsignedInt;
	    pValue->un.unsignedInt = pObject->maxAbsolute.uint32;
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;
    }

    return Success;
}


/*
 * MM_setThresholdByHandle()
 *
 * Set the maximum or minimum threshold value for a managable object.
 *
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously returned by
 *     MM_registerManagableObject().
 *
 *   thresholdType --
 *     Indication of whether the threshold to be set is a Maximum threshold or
 *     a Minimum threshold.
 *
 *   pValue --
 *     Pointer to the value to which the threshold should be set.  The
 *     value pointed to must be to the proper type for the managable
 *     object.  See the "Value Type" comments with the declaration of
 *     ManagableObjectType.
 *
 *
 * Note:
 *   Thresholds are only applicable to certain managable object types, such as
 *   Counters and Gauges.
 */
ReturnCode
MM_setThresholdByHandle(void * hManagableObject,
			MM_ThresholdType thresholdType,
			MM_Value * pValue)
{
    ManagableObject *	pObject = hManagableObject;

    /* Make sure this object type supports thresholds */
    switch(pObject->objectType)
    {
    case MM_ManagableObjectType_GaugeSigned:
    case MM_ManagableObjectType_GaugeUnsigned:
	break;

    case MM_ManagableObjectType_CounterSigned:
    case MM_ManagableObjectType_CounterUnsigned:
	if (thresholdType == MM_ThresholdType_MinimumThreshold)
	{
	    /* Minimum thresholds not supported for these types */
	    return FAIL_RC(MM_RC_MinimumThresholdNotSupported,
			   ("MM_setThreshold(MINIMUM) "
			    "not supported for type %d\n",
			    pObject->objectType));
	}
	break;
	
    case MM_ManagableObjectType_String:
    case MM_ManagableObjectType_Timer:
    case MM_ManagableObjectType_Log:
	/* Thresholds not supported for these types */
	return FAIL_RC(MM_RC_ThresholdNotSupported,
		       ("MM_setThreshold not supported for type %d\n",
			pObject->objectType));
	break;

	/* No default so the compiler will catch it if we add more types */
    }

    /* See whether we're setting the minimum or maximum threshold */
    switch(thresholdType)
    {
    case MM_ThresholdType_MinimumThreshold:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_SignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->minThreshold.sint32 = pValue->un.signedInt;
	    break;
	    
	case ValueType_Uint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_UnsignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->minThreshold.uint32 = pValue->un.unsignedInt;
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;
	
    case MM_ThresholdType_MaximumThreshold:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_SignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->maxThreshold.sint32 = pValue->un.signedInt;
	    break;
	    
	case ValueType_Uint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_UnsignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->maxThreshold.uint32 = pValue->un.unsignedInt;
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;

    case MM_ThresholdType_MinimumAbsolute:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_SignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->minAbsolute.sint32 = pValue->un.signedInt;

	    /* Make sure we're not currently exceeding the absolute minimum */
	    if (pObject->value.sint32 < pObject->minAbsolute.sint32)
	    {
		pObject->value.sint32 = pObject->minAbsolute.sint32;
	    }
	    break;
	    
	case ValueType_Uint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_UnsignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->minAbsolute.uint32 = pValue->un.unsignedInt;

	    /* Make sure we're not currently exceeding the absolute minimum */
	    if (pObject->value.uint32 < pObject->minAbsolute.uint32)
	    {
		pObject->value.uint32 = pObject->minAbsolute.uint32;
	    }
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;
	
    case MM_ThresholdType_MaximumAbsolute:
	/* Set the value. */
	switch(pObject->valueType)
	{
	case ValueType_Sint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_SignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->maxAbsolute.sint32 = pValue->un.signedInt;

	    /* Make sure we're not currently exceeding the absolute minimum */
	    if (pObject->value.sint32 > pObject->maxAbsolute.sint32)
	    {
		pObject->value.sint32 = pObject->maxAbsolute.sint32;
	    }
	    break;
	    
	case ValueType_Uint32:
	    /* Make sure the threshold type matches the object's value type */
	    if (pValue->type != MM_ValueType_UnsignedInt)
	    {
		/* Wrong value type */
		return FAIL_RC(MM_RC_WrongValueType,
			       ("MM_setThresholdByHandle(): "
				"value type does not match object type.\n"));
	    }

	    pObject->maxAbsolute.uint32 = pValue->un.unsignedInt;

	    /* Make sure we're not currently exceeding the absolute minimum */
	    if (pObject->value.uint32 > pObject->maxAbsolute.uint32)
	    {
		pObject->value.uint32 = pObject->maxAbsolute.uint32;
	    }
	    break;
	    
	default:
	    /* do nothing */
	    break;
	}
	break;
    }

    return Success;
}

	     
/*
 * MM_getValueByHandle()
 *
 * Get the current value of a managable object.
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously provided by
 *     MM_registerManagableObject().
 *
 *   pValue --
 *     Pointer to the location where the current value of the specified
 *     managable object is to be placed.  It is up to the caller to provide a
 *     pointer to the correct type of variable into which the value will be
 *     placed.
 *
 *
 * NOTE: The value of a managable object of type MM_ObjectType_String is a
 *       pointer to a COPY of the string maintained within the managable
 *       object.  It is the caller's responsibility to free this COPY, with
 *       OS_free(), when it is no longer required.
 */
ReturnCode
MM_getValueByHandle(void * hManagableObject,
		    MM_Value * pValue)
{
    ManagableObject *	pObject = hManagableObject;

    /* What type of value does this object have? */
    switch(pObject->valueType)
    {
    case ValueType_Sint32:
	pValue->type = MM_ValueType_SignedInt;
	pValue->un.signedInt = pObject->value.sint32;
	break;
	
    case ValueType_Uint32:
	pValue->type = MM_ValueType_UnsignedInt;
	pValue->un.unsignedInt = pObject->value.uint32;
	break;
	
    case ValueType_String:
	pValue->type = MM_ValueType_String;
	if ((pValue->un.string =
	     OS_allocStringCopy(pObject->value.string)) == NULL)
	{
	    return FAIL_RC(ResourceError,
			   ("MM_getValueByHandle(): "
			    "out of memory on string object\n"));
	}
	break;
	
    case ValueType_Handle:
    case ValueType_None:
	/* No value for these types */
	return FAIL_RC(MM_RC_WrongObjectType,
		       ("Attempt to get value for object type %d "
			"which has no value.\n",
			pObject->objectType));
	break;
    }

    return Success;
}


/*
 * MM_setValueByHandle()
 *
 * Set the current value of a managable object.
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously provided by
 *     MM_registerManagableObject(), for which the value is to be
 *     modified.
 *
 *   pValue --
 *     Pointer to the new value for this managable object.  It is up
 *     to the caller to provide a pointer to the correct type of
 *     variable into which the value will be placed.
 */
ReturnCode
MM_setValueByHandle(void * hManagableObject,
		    MM_Value * pValue)
{
    ManagableObject *	pObject = hManagableObject;
    char *		pOldString;
    ReturnCode		rc;

    /* What type of value does this object have? */
    switch(pObject->valueType)
    {
    case ValueType_Sint32:
	/* Make sure the new value type matches the object's value type */
	if (pValue->type != MM_ValueType_SignedInt)
	{
	    /* Wrong value type */
	    return FAIL_RC(MM_RC_WrongValueType,
			   ("MM_setValueByHandle(): "
			    "value type does not match object type.\n"));
	}

	/* See if the new value is different from the old one */
	if (pObject->value.sint32 != pValue->un.signedInt)
	{
	    /* Set the new value */
	    pObject->value.sint32 = pValue->un.signedInt;

	    /* If the value exceeds its absolute maximum... */
	    if (pObject->value.sint32 > pObject->maxAbsolute.sint32)
	    {
		/* .. set it to its maximum */
		pObject->value.sint32 = pObject->maxAbsolute.sint32;
	    }

	    /* If the value exceeds its absolute minimum... */
	    if (pObject->value.sint32 < pObject->minAbsolute.sint32)
	    {
		/* .. set it to its minimum */
		pObject->value.sint32 = pObject->minAbsolute.sint32;
	    }

	    /* If the object requests notification upon change... */
	    if (pObject->notificationTypes & MM_NotificationType_ValueChange)
	    {
		/* ... generate an event. */
		if ((rc =
		     generateEvent(MM_EventType_ValueChangedSigned, pObject,
				   0)) != Success)
		{
		    return FAIL_RC(rc, ("MM_setValueByHandle(): "
					"generate event\n"));
		}
	    }
	}

	/* If a function has been registered to call upon value change... */
	if (pObject->pfValueChanged != NULL)
	{
	    /* ... then call the registered function. */
	    (* pObject->pfValueChanged)(pObject->hUserData,
					hManagableObject, pValue);
	}

	break;
	
    case ValueType_Uint32:
	/* Make sure the new value type matches the object's value type */
	if (pValue->type != MM_ValueType_UnsignedInt)
	{
	    /* Wrong value type */
	    return FAIL_RC(MM_RC_WrongValueType,
			   ("MM_setValueByHandle(): "
			    "value type does not match object type.\n"));
	}

	/* See if the new value is different from the old one */
	if (pObject->value.uint32 != pValue->un.unsignedInt)
	{
	    /* Set the new value */
	    pObject->value.uint32 = pValue->un.unsignedInt;

	    /* If the value exceeds its absolute maximum... */
	    if (pObject->value.uint32 > pObject->maxAbsolute.uint32)
	    {
		/* .. set it to its maximum */
		pObject->value.uint32 = pObject->maxAbsolute.uint32;
	    }

	    /* If the value exceeds its absolute minimum... */
	    if (pObject->value.uint32 < pObject->minAbsolute.uint32)
	    {
		/* .. set it to its minimum */
		pObject->value.uint32 = pObject->minAbsolute.uint32;
	    }

	    /* If the object requests notification upon change... */
	    if (pObject->notificationTypes & MM_NotificationType_ValueChange)
	    {
		/* ... generate an event. */
		if ((rc =
		     generateEvent(MM_EventType_ValueChangedUnsigned, 
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc, ("MM_setValueByHandle(): "
					"generate event\n"));
		}
	    }
	}

	/* If a function has been registered to call upon value change... */
	if (pObject->pfValueChanged != NULL)
	{
	    /* ... then call the registered function. */
	    (* pObject->pfValueChanged)(pObject->hUserData,
					hManagableObject, pValue);
	}

	break;
	
    case ValueType_String:
	/* Make sure the new value type matches the object's value type */
	if (pValue->type != MM_ValueType_String)
	{
	    /* Wrong value type */
	    return FAIL_RC(MM_RC_WrongValueType,
			   ("MM_setValueByHandle(): "
			    "value type does not match object type.\n"));
	}

	/* See if the new value is different from the old one */
	if (strcmp(pObject->value.string, pValue->un.string) != 0)
	{
	    /* Save the old pointer until we make sure we succeed here */
	    pOldString = pObject->value.string;

	    /* Allocate a new string of the correct length */
	    if ((pObject->value.string =
		 OS_allocStringCopy(pValue->un.string)) == NULL)
	    {
		/* Allocation failed.  Restore old value. */
		pObject->value.string = pOldString;
		
		return FAIL_RC(ResourceError,
			       ("Out of memory "
				"allocating for string value (%s)\n",
				pValue->un.string));
	    }

	    /* Free the old string */
	    OS_free(pOldString);

	    /* If the object requests notification upon change... */
	    if (pObject->notificationTypes & MM_NotificationType_ValueChange)
	    {
		/* ... generate an event. */
		if ((rc =
		     generateEvent(MM_EventType_ValueChangedString, pObject,
				   strlen(pObject->value.string) + 1)) !=
		    Success)
		{
		    return FAIL_RC(rc, ("MM_setValueByHandle(): "
					"generate event\n"));
		}
	    }

	    /*
	     * If a function has been registered to call upon value change...
	     */
	    if (pObject->pfValueChanged != NULL)
	    {
		/* ... then call the registered function. */
		(* pObject->pfValueChanged)(pObject->hUserData,
					    hManagableObject, pValue);
	    }
	}
	break;
	
    case ValueType_Handle:
    case ValueType_None:
	/* No value for these types */
	return FAIL_RC(MM_RC_WrongObjectType,
		       ("Attempt to set value for object type %d "
			"which has no value.\n",
			pObject->objectType));
	break;
    }

    return Success;
}


/*
 * MM_incrementValue()
 *
 * Increment the numeric value of the specified managable object (probably
 * either a counter or a gauge) by the specified value.
 *
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously returned by
 *     MM_registerManagableObject().
 *
 *   incrementBy --
 *     Amount by which the managable object's value should be incremented.
 *     The increment value may be negative to decrement the value.
 */
ReturnCode
MM_incrementValue(void * hManagableObject,
		  OS_Sint32 incrementBy)
{
    ManagableObject *	pObject = hManagableObject;
    OS_Boolean		bWasWithinThresholds;
    OS_Boolean		bIsWithinThresholds;
    ReturnCode		rc;
    MM_Value		value;

    /* Make sure this object type supports increment */
    switch(pObject->objectType)
    {
    case MM_ManagableObjectType_GaugeSigned:
    case MM_ManagableObjectType_GaugeUnsigned:
    case MM_ManagableObjectType_CounterSigned:
    case MM_ManagableObjectType_CounterUnsigned:
	break;
	
    case MM_ManagableObjectType_String:
    case MM_ManagableObjectType_Timer:
    case MM_ManagableObjectType_Log:
	/* Increment not supported for these types */
	return FAIL_RC(MM_RC_IncrementNotSupported,
		       ("MM_incrementValue not supported for type %d\n",
			pObject->objectType));
	break;

	/* No default so the compiler will catch it if we add more types */
    }

    /* If we're not actually incrementing, then go away now. */
    if (incrementBy == 0)
    {
	return Success;
    }

    switch(pObject->valueType)
    {
    case ValueType_Sint32:
	/* See if we were within threshold boundaries */
	if (pObject->value.sint32 >= pObject->minThreshold.sint32 &&
	    pObject->value.sint32 <= pObject->maxThreshold.sint32)
	{
	    bWasWithinThresholds = TRUE;
	}
	else
	{
	    bWasWithinThresholds = FALSE;
	}

	/* Increment the value */
	pObject->value.sint32 += incrementBy;

	/* If the value exceeds its absolute maximum... */
	if (pObject->value.sint32 > pObject->maxAbsolute.sint32)
	{
	    /* .. set it to its maximum */
	    pObject->value.sint32 = pObject->maxAbsolute.sint32;
	}

	/* If the value exceeds its absolute minimum... */
	if (pObject->value.sint32 < pObject->minAbsolute.sint32)
	{
	    /* .. set it to its minimum */
	    pObject->value.sint32 = pObject->minAbsolute.sint32;
	}

	/* Again check for within thresholds */
	if (pObject->value.sint32 >= pObject->minThreshold.sint32 &&
	    pObject->value.sint32 <= pObject->maxThreshold.sint32)
	{
	    bIsWithinThresholds = TRUE;
	}
	else
	{
	    bIsWithinThresholds = FALSE;
	}
	
	/* If the object requests notification upon change... */
	if (pObject->notificationTypes & MM_NotificationType_ValueChange)
	{
	    /* ... generate an event. */
	    if ((rc =
		 generateEvent(MM_EventType_ValueChangedSigned, pObject,
			       0)) != Success)
	    {
		return FAIL_RC(rc, ("MM_setValueByHandle(): "
				    "generate event\n"));
	    }
	}

	/* Did we go from in-threshold to out-of-threshold or vice versa? */
	if (bWasWithinThresholds && ! bIsWithinThresholds)
	{
	    /*
	     * We went from in-threshold to out-of-threshold.
	     *
	     * Which threshold did we exceed?  We'll assume that if
	     * the incrementBy value was positive, then we exceeded
	     * the maximum threshold (even if it caused a
	     * wrap-around).  Similarly for negative values of
	     * incrementBy for the minimum threshold.
	     */
	    if (incrementBy > 0)
	    {
		if ((rc =
		     generateEvent(MM_EventType_MaxThresholdExceededSigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	    else
	    {
		if ((rc =
		     generateEvent(MM_EventType_MinThresholdExceededSigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	}
	else if (! bWasWithinThresholds && bIsWithinThresholds)
	{
	    /*
	     * We went from out-of-threshold to in-threshold.
	     *
	     * Which threshold did we reenter?  We'll assume that if
	     * the incrementBy value was positive, then we reentered
	     * the minimum threshold (even if it caused a
	     * wrap-around).  Similarly for negative values of
	     * incrementBy for the maximum threshold.
	     */
	    if (incrementBy > 0)
	    {
		if ((rc =
		     generateEvent(MM_EventType_MinThresholdReenteredSigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	    else
	    {
		if ((rc =
		     generateEvent(MM_EventType_MaxThresholdReenteredSigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	}

	/* If a function has been registered to call upon value change... */
	if (pObject->pfValueChanged != NULL)
	{
	    /* ... then create a value variable, ... */
	    value.type = MM_ValueType_SignedInt;
	    value.un.signedInt = pObject->value.sint32;

	    /* ... and call the registered function. */
	    (* pObject->pfValueChanged)(pObject->hUserData,
					hManagableObject, &value);
	}

	break;
	    
    case ValueType_Uint32:
	/* See if we were within threshold boundaries */
	if (pObject->value.uint32 >= pObject->minThreshold.uint32 &&
	    pObject->value.uint32 <= pObject->maxThreshold.uint32)
	{
	    bWasWithinThresholds = TRUE;
	}
	else
	{
	    bWasWithinThresholds = FALSE;
	}

	/* Increment the value */
	pObject->value.uint32 += incrementBy;

	/* If the value exceeds its absolute maximum... */
	if (pObject->value.uint32 > pObject->maxAbsolute.uint32)
	{
	    /* .. set it to its maximum */
	    pObject->value.uint32 = pObject->maxAbsolute.uint32;
	}

	/* If the value exceeds its absolute minimum... */
	if (pObject->value.uint32 < pObject->minAbsolute.uint32)
	{
	    /* .. set it to its minimum */
	    pObject->value.uint32 = pObject->minAbsolute.uint32;
	}

	/* Again check for within thresholds */
	if (pObject->value.uint32 >= pObject->minThreshold.uint32 &&
	    pObject->value.uint32 <= pObject->maxThreshold.uint32)
	{
	    bIsWithinThresholds = TRUE;
	}
	else
	{
	    bIsWithinThresholds = FALSE;
	}
	
	/* If the object requests notification upon change... */
	if (pObject->notificationTypes & MM_NotificationType_ValueChange)
	{
	    /* ... generate an event. */
	    if ((rc =
		 generateEvent(MM_EventType_ValueChangedUnsigned, pObject,
			       0)) != Success)
	    {
		return FAIL_RC(rc, ("MM_setValueByHandle(): "
				    "generate event\n"));
	    }
	}

	/* Did we go from in-threshold to out-of-threshold or vice versa? */
	if (bWasWithinThresholds && ! bIsWithinThresholds)
	{
	    /*
	     * We went from in-threshold to out-of-threshold.
	     *
	     * Which threshold did we exceed?  We'll assume that if
	     * the incrementBy value was positive, then we exceeded
	     * the maximum threshold (even if it caused a
	     * wrap-around).  Similarly for negative values of
	     * incrementBy for the minimum threshold.
	     */
	    if (incrementBy > 0)
	    {
		if ((rc =
		     generateEvent(MM_EventType_MaxThresholdExceededUnsigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	    else
	    {
		if ((rc =
		     generateEvent(MM_EventType_MinThresholdExceededUnsigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	}
	else if (! bWasWithinThresholds && bIsWithinThresholds)
	{
	    /*
	     * We went from out-of-threshold to in-threshold.
	     *
	     * Which threshold did we reenter?  We'll assume that if
	     * the incrementBy value was positive, then we reentered
	     * the minimum threshold (even if it caused a
	     * wrap-around).  Similarly for negative values of
	     * incrementBy for the maximum threshold.
	     */
	    if (incrementBy > 0)
	    {
		if ((rc =
		     generateEvent(MM_EventType_MinThresholdReenteredUnsigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	    else
	    {
		if ((rc =
		     generateEvent(MM_EventType_MaxThresholdReenteredUnsigned,
				   pObject, 0)) != Success)
		{
		    return FAIL_RC(rc,
				   ("MM_incrementValue(): generate event\n"));
		}
	    }
	}

	/* If a function has been registered to call upon value change... */
	if (pObject->pfValueChanged != NULL)
	{
	    /* ... then create a value variable, ... */
	    value.type = MM_ValueType_UnsignedInt;
	    value.un.unsignedInt = pObject->value.uint32;

	    /* ... and call the registered function. */
	    (* pObject->pfValueChanged)(pObject->hUserData,
					hManagableObject, &value);
	}

	break;
	    
    default:
	/* do nothing */
	break;
    }

    return Success;
}


/*
 * MM_startTimerByHandle()
 *
 * Start a timer.  When it expires, an event will be raised.
 *
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously returned by
 *     MM_registerManagableObject().
 *
 *   milliseconds --
 *     Number of milliseconds before the timer should expire.
 */
ReturnCode
MM_startTimerByHandle(void * hManagableObject,
		      OS_Uint32 milliseconds)
{
    ManagableObject *	pObject = hManagableObject;
    ReturnCode		rc;

    /* Make sure the object is a timer object */
    if (pObject->objectType != MM_ManagableObjectType_Timer)
    {
	return FAIL_RC(MM_RC_WrongObjectType,
		       ("MM_startTimer() got object type %d, not a timer.\n",
			pObject->objectType));
    }

    /* Create a timer for this object */
    if ((rc = TMR_start(milliseconds, pObject, NULL,
			timerExpiry, &pObject->value.handle)) != Success)
    {
	return FAIL_RC(rc, ("MM_startTimer(): TMR_start() failed\n"));
    }

    return Success;
}


/*
 * MM_stopTimerByHandle()
 *
 * Stop a previously started timer.
 *
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously returned by
 *     MM_registerManagableObject().
 */
ReturnCode
MM_stopTimerByHandle(void * hManagableObject)
{
    ManagableObject *	pObject = hManagableObject;

    /* Make sure the object is a timer object */
    if (pObject->objectType != MM_ManagableObjectType_Timer)
    {
	return FAIL_RC(MM_RC_WrongObjectType,
		       ("MM_startTimer() got object type %d, not a timer.\n",
			pObject->objectType));
    }

    /* Make sure there's a timer running for this object */
    if (pObject->value.handle == NULL)
    {
	/* We won't flag it as an error to stop a stopped timer. */
	return Success;
    }

    TMR_stop(pObject->value.handle);

    return Success;
}


/*
 * MM_logMessage()
 *
 * Generate a message for logging, using a printf-style format.
 *
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously returned by
 *     MM_registerManagableObject().
 *
 *   pBuf --
 *     Pointer to a buffer into which the string to be logged will be written,
 *     prior to sending the event to its destinations.  The buffer to which
 *     this pointer points must be large enough to contain the resulting
 *     string, or chaos will assuredly occur.
 *
 *   pFormat --
 *     Printf-style format string specifying the format for the remainder of
 *     the parameters.
 *
 *   ... --
 *     Additional parameters, as specified by pFormat.
 */
ReturnCode
MM_logMessage(void * hManagableObject,
	      char * pBuf,
	      char * pFormat,
	      ...)
{
    ManagableObject *	pObject = hManagableObject;
    va_list		pArgs;
    ReturnCode		rc;

    /* Make sure the managable object is a Log object */
    if (pObject->objectType != MM_ManagableObjectType_Log)
    {
	return FAIL_RC(MM_RC_WrongObjectType,
		       ("MM_logMessage() got object type %d, not a log.\n",
			pObject->objectType));
    }

    /* Get a pointer to the optional arguments */
    pArgs = va_start(pArgs, pFormat);

    /* Create the string to be logged. */
    vsprintf(pBuf, pFormat, pArgs);

    /* We're done with the optional arguments now. */
    va_end(pArgs);

    /* Assign the resulting string to this object's value */
    pObject->value.handle = pBuf;

    /* Generate a Log event */
    if ((rc = generateEvent(MM_EventType_LogMessage, pObject,
			    strlen(pBuf) + 1)) != Success)
    {
	return FAIL_RC(rc, ("MM_logMessage(): generate event\n"));
    }
    
    return Success;
}


/*
 * MM_getManagableObjectNotifyMaskByHandle()
 *
 * Get the notification mask for the specified managable object, which
 * indicates the classification and urgency of events raised on that
 * object.
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously provided by
 *     MM_registerManagableObject(), for which the notification mask
 *     is to be provided.
 *
 *   pNotifyMask --
 *     Pointer to a location in which to place the bit mask indicating
 *     which notification levels are to be sent to this destination.
 */
ReturnCode
MM_getManagableObjectNotifyMaskByHandle(void * hManagableObject,
					OS_Uint32 * pNotifyMask)
{
    ManagableObject *	pManagableObject = hManagableObject;

    /* Give 'em what they came for */
    *pNotifyMask = pManagableObject->notificationTypes;

    return Success;
}


/*
 * MM_setManagableObjectNotifyMaskByHandle()
 *
 * Set the notification mask for the specified managable object, which
 * indicates the classification and urgency of events raised on that
 * object.
 *
 * Parameters:
 *
 *   hManagableObject --
 *     Handle to a managable object, previously provided by
 *     MM_registerManagableObject(), for which the notification mask
 *     is to be provided.
 *
 *   notifyMask --
 *     New value of the bit mask indicating which notification levels
 *     are to be sent to this destination.
 */
ReturnCode
MM_setManagableObjectNotifyMaskByHandle(void * hManagableObject,
					OS_Uint32 notifyMask)
{
    ManagableObject *	pManagableObject = hManagableObject;

    /* Set the notification mask as specified. */
    pManagableObject->notificationTypes = notifyMask;

    return Success;
}


/*
 * MM_getDestinationMasksByHandle()
 *
 * Obtain the set of notification and event types which should be sent
 * to this destination.
 *
 * Parameters:
 *
 *   hDestination --
 *     Destination handle, previously provided by
 *     MM_registerDestination(), indicating the destination from which
 *     the notififcation and event masks are to be retrieved.
 *
 *   pNotifyMask --
 *     Pointer to a location in which to place the bit mask indicating
 *     which notification levels are to be sent to this destination.
 *
 *   pEventMask --
 *     Pointer to a location in which to place the bit mask indicating
 *     which events are to be sent to this destination.
 */
ReturnCode
MM_getDestinationMasksByHandle(void * hDestination,
			       OS_Uint32 * pNotifyMask,
			       OS_Uint32 * pEventMask)
{
    Destination *	pDestination = hDestination;

    /* Give 'em what they came for */
    *pNotifyMask = pDestination->notifyMask;
    *pEventMask = pDestination->eventMask;

    return Success;
}


/*
 * MM_setDestinationMasksByHandle()
 *
 * Modify the set of notification types which should be sent to this
 * destination.
 *
 *
 * Parameters:
 *
 *   hDestination --
 *     Handle, previously provided by MM_registerDestination().
 *
 *   newNotifyMask --
 *     New bit mask indicating which notification levels are to be sent to
 *     this destination.
 *
 *   newEventMask --
 *     New bit mask indicating which events are to be sent to this
 *     destination.
 */
ReturnCode
MM_setDestinationMasksByHandle(void * hDestination,
			       OS_Uint32 newNotifyMask,
			       OS_Uint32 newEventMask)
{
    Destination *	pDestination = hDestination;

    /* Set the notification mask. */
    pDestination->notifyMask = newNotifyMask;

    /* Set the event mask. */
    pDestination->eventMask = newEventMask;

    return Success;
}



/*
 * MM_processEvents()
 *
 * Event notification does not happen asynchronously.  The reason for this is
 * that the event could be raised during interrupt routines, critical
 * sections, etc.  We therefore enqueue the event notification for action when
 * this function is called.
 *
 * This function should be called on a regular basis, either in a main loop,
 * or via a timer expiration.
 *
 *
 * Parameters:
 *
 *   hApplicationEntityInstance --
 *     Handle, previously provided by MM_entityInit(), indicating to
 *     which application entity instance this module is attached.
 *
 *   pbFoundOne --
 *     Pointer to a boolean variable, which is set to TRUE by this function if
 *     an event was found to process.  This variable is _not_ modified if no
 *     event was found to process, enabling a pointer to the same variable to
 *     be passed to multiple functions to see if any of them had anything to
 *     do.
 *
 *     This pointer may be NULL if an indication of whether an event was
 *     processed is not required.
 */
ReturnCode
MM_processEvents(void * hApplicationEntityInstance,
		 OS_Boolean * pbFoundOne)
{
    Event * 	    pEvent;
    Destination *   pDestination;
    OS_Boolean	    bFoundOne = FALSE;
    ApplicationEntityInstance *	pApplicationEntityInstance;

    /* Get a pointer to the application entity instance structure */
    pApplicationEntityInstance = hApplicationEntityInstance;
    
    /* For each event in the queue... */
    for (pEvent = QU_FIRST(&pApplicationEntityInstance->pendingEvents);
	 ! QU_EQUAL(pEvent, &pApplicationEntityInstance->pendingEvents);
	 pEvent = QU_FIRST(&pApplicationEntityInstance->pendingEvents))
    {
	/* Remove the event from the queue */
	QU_REMOVE(pEvent);

	/* For each potential destination... */
	for (pDestination =
		 QU_FIRST(&pApplicationEntityInstance->destinations);
	     ! QU_EQUAL(pDestination,
			&pApplicationEntityInstance->destinations);
	     pDestination = QU_NEXT(pDestination))
	{
	    /*
	     * If the event's notification mask has bits set which are
	     * also set in this destination's notification mask, AND
	     * this destination is set to receive this event's event
	     * type, then call the destination's alert function.
	     *
	     * Some events are to be raised regardless of the object's
	     * notification mask.  Currently the following events fall
	     * into this class:
	     *
	     *         MM_EventType_ManagableObjectChange
	     */
	    if (((pEvent->notificationTypes & pDestination->notifyMask) != 0 ||
		 pEvent->eventType == MM_EventType_ManagableObjectChange) &&
		(pDestination->eventMask & (1 << pEvent->eventType)) != 0)
	    {
		/* We found an event to process.  Remember that. */
		bFoundOne = TRUE;

		/*
		 * Call destination's alert function with appropriate
		 * parameters
		 */
		switch(pEvent->eventType)
		{
		case MM_EventType_MaxThresholdExceededSigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Sint32"
		     *           - the value which caused the event by
		     *             exceeding the threshold, as an
		     *             "OS_Sint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.sint32,
					      pEvent->param2.sint32);
		    break;
	
		case MM_EventType_MaxThresholdExceededUnsigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Uint32"
		     *           - the value which caused the event by
		     *             exceeding the threshold, as an
		     *             "OS_Uint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.uint32,
					      pEvent->param2.uint32);
		    break;
	
		case MM_EventType_MinThresholdExceededSigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Sint32"
		     *           - the value which caused the event by
		     *		   exceeding the threshold, as an
		     *		   "OS_Sint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.sint32,
					      pEvent->param2.sint32);
		    break;
	
		case MM_EventType_MinThresholdExceededUnsigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Uint32"
		     *           - the value which caused the event by
		     *             exceeding the threshold, as an
		     *             "OS_Uint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.uint32,
					      pEvent->param2.uint32);
		    break;
	
		case MM_EventType_MaxThresholdReenteredSigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Sint32"
		     *           - the value which caused the event by
		     *             reentering the threshold, as an
		     *             "OS_Sint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.sint32,
					      pEvent->param2.sint32);
		    break;
	
		case MM_EventType_MaxThresholdReenteredUnsigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Uint32"
		     *           - the value which caused the event by
		     *             reentering the threshold, as an
		     *             "OS_Uint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.uint32,
					      pEvent->param2.uint32);
		    break;
	
		case MM_EventType_MinThresholdReenteredSigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Sint32"
		     *           - the value which caused the event by
		     *             reentering the threshold, as an
		     *             "OS_Sint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.sint32,
					      pEvent->param2.sint32);
		    break;
	
		case MM_EventType_MinThresholdReenteredUnsigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     Two optional parameters are passed:
		     * 
		     *           - the threshold value, as an "OS_Uint32"
		     *           - the value which caused the event by
		     *             reentering the threshold, as an
		     *             "OS_Uint32"
		     *
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.uint32,
					      pEvent->param2.uint32);
		    break;
	
		case MM_EventType_TimerExpired:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     No optional parameters are passed.
		     */    
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType);
		    break;
	
		case MM_EventType_LogMessage:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     One optional parameter is passed:
		     *
		     *		- the log message string, as a "char *"
		     */
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.string);
		    break;
		    
		case MM_EventType_ValueChangedSigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     One optional parameter is passed:
		     * 
		     *           - the new value as an "OS_Sint32"
		     */    
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.sint32);
		    break;

		case MM_EventType_ValueChangedUnsigned:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     One optional parameter is passed:
		     * 
		     *           - the new value as an "OS_Uint32"
		     */    
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.uint32);
		    break;

		case MM_EventType_ValueChangedString:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     One optional parameter is passed:
		     * 
		     *           - the new value, as a "char *"
		     */    
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType,
					      pEvent->param1.string);
		    break;

		case MM_EventType_ManagableObjectChange:
		    /*
		     * Parameters passed to the Alert function when an
		     * event of this type is raised:
		     *
		     *     No optional parameters are passed.
		     */    
		    (* pDestination->pfAlert)(pDestination->pName,
					      pApplicationEntityInstance->
					      pName,
					      pEvent->pModuleName,
					      pEvent->pObjectName,
					      pEvent->pDescription,
					      pEvent->eventType);
		    break;

		}
	    }
	}

	/* We're all done with this event.  Free it. */
	OS_free(pEvent);
    }

    /* Did we find an event to process? */
    if (bFoundOne && pbFoundOne != NULL)
    {
	/* Yup.  Let 'em know. */
	*pbFoundOne = TRUE;
    }

    return Success;
}



/*
 * mm_getApplicationEntityInstances()
 *
 * Allow the agent code to get our list of register application entity
 * instances.
 */
QU_Head *
mm_getApplicationEntityInstances(void)
{
    return &applicationEntityInstances;
}




ApplicationEntityInstance *
mm_findApplicationEntityInstance(char * pApplicationEntityInstanceName)
{
    ApplicationEntityInstance *	pApplicationEntityInstance;

    /* Find the specified application entity instance */
    for (pApplicationEntityInstance = QU_FIRST(&applicationEntityInstances);
	 ! QU_EQUAL(pApplicationEntityInstance, &applicationEntityInstances);
	 pApplicationEntityInstance = QU_NEXT(pApplicationEntityInstance))
    {
	if (strcmp(pApplicationEntityInstance->pName,
		   pApplicationEntityInstanceName) == 0)
	{
	    return pApplicationEntityInstance;
	}
    }
    
    /* Requested name found */
    return NULL;
}


Module *
mm_findModule(char * pModuleName,
	      ApplicationEntityInstance * pApplicationEntityInstance)
{
    Module * 	    pModule;

    /* Find the specified module name in the queue of modules */
    for (pModule = QU_FIRST(&pApplicationEntityInstance->modules);
	 ! QU_EQUAL(pModule, &pApplicationEntityInstance->modules);
	 pModule = QU_NEXT(pModule))
    {
	/* Is this the module we're looking for? */
	if (strcmp(pModule->pName, pModuleName) == 0)
	{
	    /* Yup.  Give 'em the pointer to this module. */
	    return pModule;
	}
    }

    /* Requested module name not found */
    return NULL;
}


ManagableObject *
mm_findObject(char * pManagableObjectName,
	      Module * pModule)
{
    ManagableObject * 		pObject;

    /* Find the specified object name in the queue of managable objects */
    for (pObject = QU_FIRST(&pModule->managableObjects);
	 ! QU_EQUAL(pObject, &pModule->managableObjects);
	 pObject = QU_NEXT(pObject))
    {
	/* Is this the managable object we're looking for? */
	if (strcmp(pObject->pName, pManagableObjectName) == 0)
	{
	    /* Yup.  Give 'em the pointer to this object. */
	    return pObject;
	}
    }
    
    /* Requested object not found. */
    return NULL;
}


Destination *
mm_findDestination(char * pDestinationName,
		   ApplicationEntityInstance * pApplicationEntityInstance)
{
    Destination *	pDestination;

    /* Find the specified destination name in the queue of destinations */
    for (pDestination = QU_FIRST(&pApplicationEntityInstance->destinations);
	 ! QU_EQUAL(pDestination, &pApplicationEntityInstance->destinations);
	 pDestination = QU_NEXT(pDestination))
    {
	/* Is this the destination we're looking for? */
	if (strcmp(pDestination->pName, pDestinationName) == 0)
	{
	    /* Yup.  Give 'em the pointer to the destination. */
	    return pDestination;
	}
    }

    /* Requested destination not found. */
    return NULL;
}

static ReturnCode
generateEvent(MM_EventType eventType,
	      ManagableObject * pObject,
	      OS_Uint16 extraMemory)
{
    Event * 			pEvent;
    Module *			pModule;
    ApplicationEntityInstance *	pApplicationEntityInstance;

    /* Get the module pointer from the object */
    pModule = pObject->pModule;

    /* Get the application entity instance from the object/module */
    pApplicationEntityInstance = pModule->pApplicationEntityInstance;

    /*
     * If there are no destinations registered, don't bother
     * generating an event.
     */
    if (QU_EQUAL(QU_FIRST(&pApplicationEntityInstance->destinations),
		 &pApplicationEntityInstance->destinations))
    {
	return Success;
    }

    /*
     * Special case.  If the event type is ManagableObjectChange, we only
     * generate one of these.  Go through the event list and if we find an
     * existing ManagableObjectChange event, we have nothing more to do.
     */
    if (eventType == MM_EventType_ManagableObjectChange)
    {
	/* Looking at each event in the queue... */
	for (pEvent =
		 QU_FIRST(&pApplicationEntityInstance->pendingEvents);
	     ! QU_EQUAL(pEvent,
			&pApplicationEntityInstance->pendingEvents);
	     pEvent = QU_NEXT(pEvent))
	{
	    /* ... if the event type is ManagableObjectChange, ... */
	    if (pEvent->eventType ==
		MM_EventType_ManagableObjectChange)
	    {
		/* ... then we have nothing more to do. */
		return Success;
	    }
	}
    }

    /* Allocate an event element */
    if ((pEvent = OS_alloc(sizeof(Event) + extraMemory)) == NULL)
    {
	return FAIL_RC(ResourceError, ("generateEvent() out of memory\n"));
    }

    /* Save the event type */
    pEvent->eventType = eventType;

    /* Save the module's name */
    pEvent->pModuleName = pModule->pName;

    /* Save the object's name */
    pEvent->pObjectName = pObject->pName;

    /* Save the othect's id message */
    pEvent->pDescription = pObject->pDescription;

    /* Save the object's notification mask */
    pEvent->notificationTypes = pObject->notificationTypes;

    /* Which event was generated? */
    switch(eventType)
    {
    case MM_EventType_MaxThresholdExceededSigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Sint32"
	 *           - the value which caused the event by exceeding the
	 *             threshold, as an "OS_Sint32"
	 *
	 */
	pEvent->param1.sint32 = pObject->maxThreshold.sint32;
	pEvent->param2.sint32 = pObject->value.sint32;
	break;
	
    case MM_EventType_MaxThresholdExceededUnsigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Uint32"
	 *           - the value which caused the event by exceeding the
	 *             threshold, as an "OS_Uint32"
	 *
	 */
	pEvent->param1.uint32 = pObject->maxThreshold.uint32;
	pEvent->param2.uint32 = pObject->value.uint32;
	break;
	
    case MM_EventType_MinThresholdExceededSigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Sint32"
	 *           - the value which caused the event by exceeding the
	 *             threshold, as an "OS_Sint32"
	 *
	 */
	pEvent->param1.sint32 = pObject->minThreshold.sint32;
	pEvent->param2.sint32 = pObject->value.sint32;
	break;
	
    case MM_EventType_MinThresholdExceededUnsigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Uint32"
	 *           - the value which caused the event by exceeding the
	 *             threshold, as an "OS_Uint32"
	 *
	 */
	pEvent->param1.uint32 = pObject->minThreshold.uint32;
	pEvent->param2.uint32 = pObject->value.uint32;
	break;
	
    case MM_EventType_MaxThresholdReenteredSigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Sint32"
	 *           - the value which caused the event by reentering the
	 *             threshold, as an "OS_Sint32"
	 *
	 */
	pEvent->param1.sint32 = pObject->maxThreshold.sint32;
	pEvent->param2.sint32 = pObject->value.sint32;
	break;
	
    case MM_EventType_MaxThresholdReenteredUnsigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Uint32"
	 *           - the value which caused the event by reentering the
	 *             threshold, as an "OS_Uint32"
	 *
	 */
	pEvent->param1.uint32 = pObject->maxThreshold.uint32;
	pEvent->param2.uint32 = pObject->value.uint32;
	break;
	
    case MM_EventType_MinThresholdReenteredSigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Sint32"
	 *           - the value which caused the event by reentering the
	 *             threshold, as an "OS_Sint32"
	 *
	 */
	pEvent->param1.sint32 = pObject->minThreshold.sint32;
	pEvent->param2.sint32 = pObject->value.sint32;
	break;
	
    case MM_EventType_MinThresholdReenteredUnsigned:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     Two optional parameters are passed:
	 * 
	 *           - the threshold value, as an "OS_Uint32"
	 *           - the value which caused the event by reentering the
	 *             threshold, as an "OS_Uint32"
	 *
	 */
	pEvent->param1.uint32 = pObject->minThreshold.uint32;
	pEvent->param2.uint32 = pObject->value.uint32;
	break;
	
    case MM_EventType_TimerExpired:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     No optional parameters are passed.
	 */    
	break;
	
    case MM_EventType_LogMessage:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     One optional parameter is passed:
	 *
	 *		- the log message string, as a "char *"
	 */

	/* Copy the message since it's currently in the caller's buffer */
	pEvent->param1.string = (char *) (pEvent + 1);
	strcpy(pEvent->param1.string, (char *) pObject->value.handle);
	break;

    case MM_EventType_ValueChangedSigned:
	/*
	 * Parameters passed to the Alert function when an
	 * event of this type is raised:
	 *
	 *     One optional parameter is passed:
	 * 
	 *           - the new value as an "OS_Sint32"
	 */    
	pEvent->param1.sint32 = pObject->value.sint32;
	break;
	
    case MM_EventType_ValueChangedUnsigned:
	/*
	 * Parameters passed to the Alert function when an
	 * event of this type is raised:
	 *
	 *     One optional parameter is passed:
	 * 
	 *           - the new value as an "OS_Uint32"
	 */    
	pEvent->param1.uint32 = pObject->value.uint32;
	break;
	
    case MM_EventType_ValueChangedString:
	/*
	 * Parameters passed to the Alert function when an
	 * event of this type is raised:
	 *
	 *     One optional parameter is passed:
	 * 
	 *           - the new value, as a "char *"
	 */    
	/* Copy the message since it's currently in the caller's buffer */
	pEvent->param1.string = (char *) (pEvent + 1);
	strcpy(pEvent->param1.string, (char *) pObject->value.string);
	break;
	
    case MM_EventType_ManagableObjectChange:
	/*
	 * Parameters passed to the Alert function when an event of this
	 * type is raised:
	 *
	 *     No optional parameters are passed.
	 *
	 * This event is only generated once per call to
	 * MM_processEvents().  This allows for a number of managable
	 * objects to be created, with only one event telling the
	 * manager application to re-read its list of managable object
	 * names.
	 */    
	break;
    }

    /* Initialize the queue pointers */
    QU_INIT(pEvent);

    /* Insert this event onto the event queue */
    QU_INSERT(pEvent, &pApplicationEntityInstance->pendingEvents);

    return Success;
}


static ReturnCode
timerExpiry(void * hTimer,
	    void * hUserData1,
	    void * hUserData2)
{
    ManagableObject *	pObject;
    ReturnCode		rc;

    /* Get the object which generated this timer expiry */
    pObject = hUserData1;

    /*
     * Clean up so we don't later think there's a timer active for
     * this object.
     */
    pObject->value.handle = NULL;

    /* Generate a timer-expiry event */
    if ((rc = generateEvent(MM_EventType_TimerExpired, pObject, 0)) != Success)
    {
	return FAIL_RC(rc, ("timerExpiry(): generate event\n"));
    }

    return Success;
}
